<?php

namespace App\Http\Controllers;

use App\Exports\EstudianteExport;
use App\Imports\EstudianteImport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class EstudianteController extends Controller
{

    public function index()
    {

        $datosEstudiante = DB::table('estudiante')
            ->join('padre_familia', 'estudiante.id_padre_familia', '=', 'padre_familia.id_padre_familia')
            ->select(
                'estudiante.*',
                'padre_familia.padre_nombres',
                'padre_familia.padre_dni',
                'padre_familia.padre_ape_pat',
                'padre_familia.padre_ape_mat'
            )
            ->orderBy('estudiante.nivel', 'asc')
            ->orderBy('estudiante.grado', 'asc')
            ->orderBy('estudiante.ape_pat', 'asc')
            ->orderBy('estudiante.ape_mat', 'asc')
            ->orderBy('estudiante.nombres', 'asc')
            ->paginate(10);

        $padreFamilia = DB::select('select * from padre_familia order by padre_ape_pat asc');

        return view('vistas/estudiantes.index', compact("datosEstudiante"))->with('padreFamilia', $padreFamilia);
    }


    public function create()
    {
        //
    }


    public function store(Request $request)
    {
        $request->validate(
            [
                'txtnombre' => 'required',
                'txtapellidopaterno' => 'required',
                'txtapellidomaterno' => 'required',
                'txtdni' => 'required',
                'txtpadrefamilia' => 'required',
                'txtnivel' => 'required',
                'txtgrado' => 'required',
            ]
        );
        //validar que el dni no se repita
        $dni = $request->txtdni;
        $dniRepetido = DB::select('select * from estudiante where dni = ?', [$dni]);
        if ($dniRepetido) {
            return redirect()->route('estudiantes.index')->with('INCORRECTO', 'El DNI ingresado ya existe');
        } else {
            //insertar datos
            $nombre = $request->txtnombre;
            $apellidoPaterno = $request->txtapellidopaterno;
            $apellidoMaterno = $request->txtapellidomaterno;
            $dni = $request->txtdni;
            $padreFamilia = $request->txtpadrefamilia;

            $res = DB::insert('insert into estudiante (id_padre_familia, dni, nombres, ape_pat, ape_mat, nivel, grado) values (?, ?, ?, ?, ?,?,?)', [$padreFamilia, $dni, $nombre, $apellidoPaterno, $apellidoMaterno, $request->txtnivel, $request->txtgrado]);
            if ($res) {
                return redirect()->route('estudiantes.index')->with('CORRECTO', 'Estudiante registrado correctamente');
            } else {
                return redirect()->route('estudiantes.index')->with('INCORRECTO', 'Error al registrar estudiante');
            }
        }
    }


    public function show($id)
    {
        $datosEstudiante = DB::select("SELECT
        estudiante.*,
        padre_familia.padre_dni,
        padre_familia.padre_nombres,
        padre_familia.padre_ape_pat,
        padre_familia.padre_ape_mat
        FROM
        estudiante
        INNER JOIN padre_familia ON estudiante.id_padre_familia = padre_familia.id_padre_familia
        where id_estudiante=$id");

        $datosPadreFamilia = DB::select("select * from padre_familia order by padre_ape_pat asc");

        return view('vistas/estudiantes.show', compact("datosEstudiante"))->with('datosPadreFamilia', $datosPadreFamilia);
    }


    public function edit($id)
    {
        //
    }


    public function update(Request $request, $id)
    {
        $request->validate([
            'txtnombre' => 'required',
            'txtapellidopaterno' => 'required',
            'txtapellidomaterno' => 'required',
            'txtdni' => 'required',
            'txtpadrefamilia' => 'required',
            'txtnivel' => 'required',
            'txtgrado' => 'required',
        ]);

        //validar que el dni no se repita
        $dni = $request->txtdni;
        $dniRepetido = DB::select("select * from estudiante where dni = ? and id_estudiante!=$id", [$dni]);
        if ($dniRepetido) {
            return redirect()->back()->with('INCORRECTO', 'El DNI ingresado ya existe');
        } else {
            //actualizar datos
            $nombre = $request->txtnombre;
            $apellidoPaterno = $request->txtapellidopaterno;
            $apellidoMaterno = $request->txtapellidomaterno;
            $dni = $request->txtdni;
            $padreFamilia = $request->txtpadrefamilia;
            $nivel = $request->txtnivel;
            $grado = $request->txtgrado;

            try {
                $res = DB::update('update estudiante set id_padre_familia = ?, dni = ?, nombres = ?, ape_pat = ?, ape_mat = ?, nivel=?, grado=? where id_estudiante = ?', [$padreFamilia, $dni, $nombre, $apellidoPaterno, $apellidoMaterno, $nivel, $grado, $id]);
                if ($res == 0) {
                    $res = 1;
                }
            } catch (\Throwable $th) {
                $res = 0;
            }
            if ($res == 1) {
                return redirect()->back()->with('CORRECTO', 'Estudiante actualizado correctamente');
            } else {
                return redirect()->back()->with('INCORRECTO', 'Error al actualizar estudiante');
            }
        }
    }


    public function destroy($id)
    {
        $res = DB::delete('delete from estudiante where id_estudiante = ?', [$id]);
        if ($res) {
            return redirect()->route('estudiantes.index')->with('CORRECTO', 'Estudiante eliminado correctamente');
        } else {
            return redirect()->route('estudiantes.index')->with('INCORRECTO', 'Error al eliminar estudiante');
        }
    }

    public function vacearRegistro()
    {
        $res = DB::delete('delete from estudiante');
        if ($res) {
            return redirect()->route('estudiantes.index')->with('CORRECTO', 'Estudiantes eliminados correctamente');
        } else {
            return redirect()->route('estudiantes.index')->with('INCORRECTO', 'Error al eliminar estudiantes');
        }
    }

    public function modeloEstudianteExcel()
    {
        return Excel::download(new EstudianteExport, 'modeloEstudiante.xlsx'); //csv
    }

    public function importarEstudiantes(Request $request)
    {
        $request->validate([
            'txtfile' => 'required|file|mimes:xlsx,xls,csv'
        ]);
        //validar que el nombre del archivo sea modeloEstudiante
        $nombreArchivo = $request->file('txtfile')->getClientOriginalName();
        if ($nombreArchivo != "modeloEstudiante.xlsx") {
            return redirect()->route('estudiantes.index')->with('INCORRECTO', 'El archivo debe llamarse modeloEstudiante');
        }
        
        $file = $request->file("txtfile"); //dato es el name
        if (Excel::import(new EstudianteImport(), $file)) {
            return redirect()->route('estudiantes.index')->with('CORRECTO', 'Estudiantes importados correctamente');
        } else {
            return redirect()->route('estudiantes.index')->with('INCORRECTO', 'Error al importar estudiantes');
        }
    }

    public function buscarEstudiante(Request $request)
    {

        $valor = $request->txtdni;
        if ($valor == null) {
            return response()->json([
                "success" => false,
                "data" => []
            ], 400);
        }

        $datosEstudiante = DB::select("SELECT
        estudiante.*,
        padre_familia.padre_dni,
        padre_familia.padre_nombres,
        padre_familia.padre_ape_pat,
        padre_familia.padre_ape_mat
        FROM
        estudiante
        INNER JOIN padre_familia ON estudiante.id_padre_familia = padre_familia.id_padre_familia where dni like '%$valor%' or nombres like '%$valor%' or 
        ape_pat like '%$valor%' or ape_mat like '%$valor%' or padre_dni like '%$valor%' limit 5");

        if ($datosEstudiante) {
            return response()->json([
                "success" => true,
                "data" => $datosEstudiante
            ], 200);
        } else {
            return response()->json([
                "success" => false,
                "data" => []
            ], 400);
        }
    }

    public function exportarEstudiantes()
    {
        return Excel::download(new EstudianteExport, 'estudiantes.xlsx'); //csv
    }

    public function actualizarGradosEstudiantes(Request $request)
    {
        $request->validate([
            'txtfile' => 'required|file|mimes:xlsx,xls,csv'
        ]);
        
        // Validar que el archivo tenga el formato correcto
        try {
            $file = $request->file("txtfile");
            $array = Excel::toArray([], $file);
            
            $actualizados = 0;
            $errores = 0;
            $sinCambios = 0;
            $noEncontrados = 0;
            $detalles = [];
            
            // Información de depuración
            $detalles[] = "Total de filas en el Excel: " . count($array[0]);
            
            // Verificar la estructura del Excel y encontrar las columnas correctas
            $dniColumnIndex = null;
            $gradoColumnIndex = null;
            $nivelColumnIndex = null;
            
            // Buscar columnas en la primera fila (encabezados)
            if (count($array[0]) > 0) {
                foreach ($array[0][0] as $index => $value) {
                    // Convertir a minúsculas para comparación insensible a mayúsculas/minúsculas
                    $columnName = strtolower(trim($value));
                    if ($columnName === 'dni') {
                        $dniColumnIndex = $index;
                    } else if ($columnName === 'grado') {
                        $gradoColumnIndex = $index;
                    } else if ($columnName === 'nivel') {
                        $nivelColumnIndex = $index;
                    }
                }
                
                $detalles[] = "Índice de columna DNI: " . ($dniColumnIndex !== null ? $dniColumnIndex : "No encontrado");
                $detalles[] = "Índice de columna Grado: " . ($gradoColumnIndex !== null ? $gradoColumnIndex : "No encontrado");
                $detalles[] = "Índice de columna Nivel: " . ($nivelColumnIndex !== null ? $nivelColumnIndex : "No encontrado");
            }
            
            // Si no se encuentran las columnas, intentar con las claves directamente
            if ($dniColumnIndex === null || ($gradoColumnIndex === null && $nivelColumnIndex === null)) {
                // Mostrar las claves disponibles para depuración
                if (count($array[0]) > 0) {
                    $detalles[] = "Claves disponibles en la primera fila: " . implode(", ", array_keys($array[0][0]));
                    
                    // Intentar encontrar por claves directamente
                    if (isset($array[0][0]['dni'])) {
                        $dniColumnIndex = 'dni';
                    }
                    
                    if (isset($array[0][0]['grado'])) {
                        $gradoColumnIndex = 'grado';
                    }
                    
                    if (isset($array[0][0]['nivel'])) {
                        $nivelColumnIndex = 'nivel';
                    }
                }
            }
            
            // Si todavía no se encuentran, intentar buscar columnas que contengan los términos
            if ($dniColumnIndex === null || ($gradoColumnIndex === null && $nivelColumnIndex === null)) {
                if (count($array[0]) > 0) {
                    foreach (array_keys($array[0][0]) as $key) {
                        $keyLower = strtolower($key);
                        if ($dniColumnIndex === null && strpos($keyLower, 'dni') !== false) {
                            $dniColumnIndex = $key;
                        }
                        if ($gradoColumnIndex === null && strpos($keyLower, 'grado') !== false) {
                            $gradoColumnIndex = $key;
                        }
                        if ($nivelColumnIndex === null && strpos($keyLower, 'nivel') !== false) {
                            $nivelColumnIndex = $key;
                        }
                    }
                }
            }
            
            // Verificar si se encontraron las columnas necesarias
            if ($dniColumnIndex === null) {
                return redirect()->route('estudiantes.index')
                    ->with('INCORRECTO', 'No se encontró la columna "dni" en el archivo Excel. Por favor, verifique el formato del archivo.');
            }
            
            if ($gradoColumnIndex === null && $nivelColumnIndex === null) {
                return redirect()->route('estudiantes.index')
                    ->with('INCORRECTO', 'No se encontró ninguna columna "grado" o "nivel" en el archivo Excel. Por favor, verifique el formato del archivo.');
            }
            
            // Procesar el Excel
            foreach ($array[0] as $index => $row) {
                // Saltar la primera fila (encabezados)
                if ($index === 0) {
                    continue;
                }
                
                $dni = is_numeric($dniColumnIndex) ? $row[$dniColumnIndex] : $row[$dniColumnIndex] ?? null;
                $grado = $gradoColumnIndex !== null ? (is_numeric($gradoColumnIndex) ? $row[$gradoColumnIndex] : $row[$gradoColumnIndex] ?? null) : null;
                $nivel = $nivelColumnIndex !== null ? (is_numeric($nivelColumnIndex) ? $row[$nivelColumnIndex] : $row[$nivelColumnIndex] ?? null) : null;
                
                // Verificar que al menos el DNI no sea nulo
                if (!empty($dni)) {
                    // Primero verificamos si el estudiante existe
                    $estudiante = DB::table('estudiante')
                        ->where('dni', $dni)
                        ->first();
                    
                    if ($estudiante) {
                        $updateData = [];
                        $cambios = [];
                        
                        // Agregar grado a los datos de actualización si está disponible
                        if (!empty($grado) && $estudiante->grado != $grado) {
                            $updateData['grado'] = $grado;
                            $cambios[] = "grado de '{$estudiante->grado}' a '{$grado}'";
                        }
                        
                        // Agregar nivel a los datos de actualización si está disponible
                        if (!empty($nivel) && $estudiante->nivel != $nivel) {
                            $updateData['nivel'] = $nivel;
                            $cambios[] = "nivel de '{$estudiante->nivel}' a '{$nivel}'";
                        }
                        
                        // Si hay cambios, actualizar el registro
                        if (!empty($updateData)) {
                            $resultado = DB::table('estudiante')
                                ->where('dni', $dni)
                                ->update($updateData);
                            
                            $actualizados++;
                            $detalles[] = "DNI: {$dni} - Actualizado: " . implode(', ', $cambios);
                        } else {
                            // No hay cambios
                            $sinCambios++;
                            $detalles[] = "DNI: {$dni} - Sin cambios, todos los datos son iguales";
                        }
                    } else {
                        // El estudiante no existe
                        $noEncontrados++;
                        $detalles[] = "DNI: {$dni} - No encontrado en el sistema";
                    }
                } else {
                    // Saltamos filas sin DNI válido
                    $errores++;
                    $detalles[] = "Fila " . ($index + 1) . " - Datos incompletos: DNI vacío";
                }
            }
            
            // Guardamos los detalles en sesión para depuración
            session(['actualizacion_detalles' => $detalles]);
            
            // Preparamos el mensaje de respuesta
            $mensaje = "Procesamiento completado:\n";
            $mensaje .= "- Registros actualizados: $actualizados\n";
            $mensaje .= "- Registros sin cambios: $sinCambios\n";
            $mensaje .= "- DNIs no encontrados: $noEncontrados\n";
            $mensaje .= "- Errores de formato: $errores\n";
            
            if ($actualizados > 0) {
                return redirect()->route('estudiantes.index')
                    ->with('CORRECTO', "Se actualizaron $actualizados estudiantes correctamente");
            } else if ($sinCambios > 0 && $actualizados == 0) {
                return redirect()->route('estudiantes.index')
                    ->with('INCORRECTO', "No se realizaron actualizaciones. Todos los datos ya estaban actualizados ($sinCambios registros sin cambios)");
            } else {
                return redirect()->route('estudiantes.index')
                    ->with('INCORRECTO', "No se pudo actualizar ningún registro. $mensaje");
            }
        } catch (\Exception $e) {
            return redirect()->route('estudiantes.index')
                ->with('INCORRECTO', 'Error al procesar el archivo: ' . $e->getMessage());
        }
    }

    /**
     * Cambia el estado de un estudiante entre activo e inactivo
     */
    public function toggleStatus($id)
    {
        try {
            // Obtener el estudiante
            $estudiante = DB::table('estudiante')->where('id_estudiante', $id)->first();
            
            if (!$estudiante) {
                return response()->json([
                    'success' => false,
                    'message' => 'Estudiante no encontrado'
                ]);
            }
            
            // Determinar el nuevo estado
            $nuevoStatus = property_exists($estudiante, 'estado') ? 
                            ($estudiante->estado == 1 ? 0 : 1) : 
                            0; // Si no existe, asumimos que estaba activo y lo desactivamos
            
            // Actualizar el estado
            $res = DB::table('estudiante')
                    ->where('id_estudiante', $id)
                    ->update(['estado' => $nuevoStatus]);
            
            if ($res) {
                $mensaje = $nuevoStatus == 1 ? 
                            'Estudiante activado correctamente' : 
                            'Estudiante desactivado correctamente';
                            
                return response()->json([
                    'success' => true,
                    'message' => $mensaje,
                    'status' => $nuevoStatus
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Error al cambiar el estado del estudiante'
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }
}
